#!/bin/bash


printf "\n\n\n\n**** RUNNING BUILD ********************\n\n"
APPDATE=`date "+%B %d, %Y"`

if [ -z $APP_NAME ] || [ -z $APP_VERSION ]; then
    source app.env
fi

if [ -z $BUILD_NUMBER ]; then
    BUILD_NUMBER="SNAPSHOT"
fi

if [ -z $JOB_NAME ]; then
    JOB_NAME=$APP_NAME
fi

if [ -z $STAGING_DIR ]; then
    printf "STAGING_DIR not specified, defaulting to dist/\n";
    STAGING_DIR="dist"
else
    STAGING_DIR=$(echo ${STAGING_DIR} | sed 's:/*$::')
    printf "Using staging directory '$STAGING_DIR'\n";
fi

if [ -d $STAGING_DIR ] && [ $STAGING_DIR != "." ]; then
    printf "Cleaning staging directory...\n"
    rm -r $STAGING_DIR
else
    printf "No staging directory to be cleaned at ${STAGING_DIR}\n"
fi

if [ -z $BUILD_DIR ]; then
    printf "Artifact directory 'BUILD_DIR' not specified, defaulting to build\n";
    BUILD_DIR="build"
else
    BUILD_DIR=$(echo ${BUILD_DIR} | sed 's:/*$::')
    printf "Using build directory '$BUILD_DIR'\n";
fi

if [ -d $BUILD_DIR ] && [ $BUILD_DIR != "." ]; then
    printf "Cleaning build directory...\n"
    rm -r $BUILD_DIR
else
    printf "No build directory to be cleaned at ${BUILD_DIR}\n"
fi

if [ -z ${DTR_PREFIX// }${DTR_ORG// } ]; then
    printf "Both DTR_PREFIX and DTR_ORG not set, pointing Docker images to dev\n"
    DTR_PREFIX=dev
fi

if [ -f $JOB_NAME*.tar.gz ]; then
    printf "Cleaning old build artifacts from workspace root\n"
    rm $JOB_NAME*.tar.gz
fi

cp gradle.properties gradle.properties.bak
cp VeteranAppointmentRequestService/build.gradle VeteranAppointmentRequestService/build.gradle.bak
cp VeteranAppointmentRequestService/src/main/webapp/WEB-INF/web.xml VeteranAppointmentRequestService/src/main/webapp/WEB-INF/web.xml.bak
/usr/bin/perl -i -pe "s|%%APP_VERSION%%|$APP_VERSION|" gradle.properties || { echo "FATAL: Could not set App Version"; exit 1; }
/usr/bin/perl -i -pe "s|%%APP_CONTEXT_VERSION%%|$APP_CONTEXT_VERSION|" VeteranAppointmentRequestService/build.gradle || { echo "FATAL: Could not set App Context Version"; exit 1; }
/usr/bin/perl -i -pe "s|%%APP_CONTEXT_VERSION%%|$APP_CONTEXT_VERSION|" VeteranAppointmentRequestService/src/main/webapp/WEB-INF/web.xml || { echo "FATAL: Could not set App Context Version"; exit 1; }

function on_fail(){
        echo "FATAL: Gradle build failed -- can't continue";
        echo "BUILD FAILED exit code $1"
        # restore the changed files
        mv gradle.properties.bak gradle.properties
        mv VeteranAppointmentRequestService/build.gradle.bak VeteranAppointmentRequestService/build.gradle
        mv VeteranAppointmentRequestService/src/main/webapp/WEB-INF/web.xml.bak VeteranAppointmentRequestService/src/main/webapp/WEB-INF/web.xml

        exit $1;
}

function build_skip_tests(){
    ./gradlew clean build -x test || on_fail 1;
}
function build_with_tests(){
    build_skip_tests;
    ./gradlew test || on_fail 2;
}
function tests(){
# runs the tests but will auto succeed which can be useful along with the --dev command to build and see how unit tests are doing without actually getting a build failed.
    ./gradlew --continue test
}

if [[ "$#" -eq 0 ]]; then
    build_with_tests;
else
    while [[ "$#" > 0 ]]; do
        case $1 in
            # just test it
            --test|--test-mode) tests; shift;;
            # just build it
            --dev|--dev-mode) build_skip_tests; shift;;
            *) echo "Unknown parameter: $1"; shift;;
        esac;
    done
fi

mv gradle.properties.bak gradle.properties
mv VeteranAppointmentRequestService/build.gradle.bak VeteranAppointmentRequestService/build.gradle
mv VeteranAppointmentRequestService/src/main/webapp/WEB-INF/web.xml.bak VeteranAppointmentRequestService/src/main/webapp/WEB-INF/web.xml

mkdir -p $BUILD_DIR
mkdir -p $STAGING_DIR
cp app.env $STAGING_DIR
cp app.yml $STAGING_DIR
cp docker-compose-fortify.yml $STAGING_DIR
cp docker-compose.yml $STAGING_DIR
cp docker-compose-beta.yml $STAGING_DIR
cp Dockerfile $STAGING_DIR
cp VeteranAppointmentRequestService/build/libs/*.war $STAGING_DIR
cp ROOT.xml $STAGING_DIR
cp entrypoint.sh $STAGING_DIR

# Set DTR for Docker - Perform against ALL Dockerfiles in your project
/usr/bin/perl -i -pe "s|%%DTR_PREFIX%%|$DTR_PREFIX|g" $STAGING_DIR/Dockerfile || { echo "FATAL: Could not set DTR Prefix"; exit 1; }
/usr/bin/perl -i -pe "s|%%DTR_ORG%%|$DTR_ORG|g" $STAGING_DIR/Dockerfile || { echo "FATAL: Could not set DTR Organization'"; exit 1; }
/usr/bin/perl -i -pe "s|%%APP_VERSION%%|$APP_VERSION|g" $STAGING_DIR/Dockerfile || { echo "FATAL: Could not set App Version"; exit 1; }
/usr/bin/perl -i -pe "s|%%APP_CONTEXT_VERSION%%|$APP_CONTEXT_VERSION|g" $STAGING_DIR/Dockerfile || { echo "FATAL: Could not set App Context Version"; exit 1; }

# Build Artifact Production
printf "\n\n**** Optional: Producing Build Artifacts ********************\n\n"

ARTIFACT="${BUILD_DIR}/${JOB_NAME}.BUILD-${BUILD_NUMBER}.tar.gz"
printf "Building application artifact ${ARTIFACT}...\n\n"
tar -C $STAGING_DIR -zcvf $ARTIFACT . || { echo "FATAL: Failed on 'Artifact tar''"; exit 1; }
cp $ARTIFACT .

printf "\n\n\n\n**** COMPLETED BUILD ********************\n\n"
